/*
 * Optimised ANSI C code for the AES cipher AKA Rijndael
 * authors: v1.0: Antoon Bosselaers
 *          v2.0: Vincent Rijmen
 * Formatted for PGP.
 * $Id: pgpAES.c,v 1.6.2.1 2001/06/21 22:33:28 hal Exp $

 */

#include "pgpSDKBuildFlags.h"

#ifndef PGP_AES
#error you must define PGP_AES one way or the other
#endif

#if PGP_AES	/* [ */

#include "pgpConfig.h"

#include "pgpSymmetricCipherPriv.h"
#include "pgpAES.h"
#include "pgpMem.h"
#include "pgpUsuals.h"

#include <stdio.h>
#include <stdlib.h>

#define MAXKC				(256/32)
#define MAXROUNDS			14

typedef unsigned char		word8;	
typedef unsigned short		word16;	
typedef unsigned int		word32;

static int ROUNDS;

static int rijndaelKeySched (word8 k[MAXKC][4], int keyBits,  
		word8 rk[MAXROUNDS+1][4][4]);
static int rijndaelEncrypt (word8 a[16], word8 b[16], 
		word8 rk[MAXROUNDS+1][4][4]);
static int rijndaelKeyEnctoDec (int keyBits, word8 W[MAXROUNDS+1][4][4]);
static int rijndaelKeyDectoEnc (int keyBits, word8 W[MAXROUNDS+1][4][4]);
static int rijndaelDecrypt (word8 a[16], word8 b[16],
		word8 rk[MAXROUNDS+1][4][4]);


#define SC	((BC - 4) >> 1)

#include "pgpAESboxes.h"
#if ! PGP_OSX	/*	These do not appear to be used.	*/
static word8 shifts[3][4][2] = {
   { { 0, 0 },
     { 1, 3 },
     { 2, 2 },
     { 3, 1 } },
   
   { { 0, 0 },
     { 1, 5 },
     { 2, 4 },
     { 3, 3 } },
   
   { { 0, 0 },
     { 1, 7 },
     { 3, 5 },
     { 4, 4 } }
}; 
#endif

static word8 mul(word8 a, word8 b) {
   /* multiply two elements of GF(2^m)
    * needed for MixColumn and InvMixColumn
    */
	if (a && b)
		return Alogtable[(Logtable[a] + Logtable[b])%255];
	else
		return 0;
}

#if ! PGP_OSX	/*	These do not appear to be used	*/
static void KeyAddition(word8 a[4][4], word8 rk[4][4], word8 BC) {
	/* Exor corresponding text input and round key input bytes
	 */
	int i, j;
	
	for(i = 0; i < BC; i++)
   	for(j = 0; j < 4; j++)
			a[i][j] ^= rk[i][j];
}

static void ShiftRow(word8 a[4][4], word8 d, word8 BC) {
	/* Row 0 remains unchanged
	 * The other three rows are shifted a variable amount
	 */
	word8 tmp[4];
	int i, j;
	
	for(i = 1; i < 4; i++) {
		for(j = 0; j < BC; j++)
			tmp[j] = a[(j + shifts[SC][i][d]) % BC][i];
		for(j = 0; j < BC; j++)
			a[j][i] = tmp[j];
	}
}

static void Substitution(word8 a[4][4], word8 box[256], word8 BC) {
	/* Replace every byte of the input by the byte at that place
	 * in the nonlinear S-box
	 */
	int i, j;
	
	for(i = 0; i < BC; i++)
		for(j = 0; j < 4; j++)
			a[i][j] = box[a[i][j]] ;
}
#endif
   
static void MixColumn(word8 a[4][4], word8 BC) {
        /* Mix the four bytes of every column in a linear way
	 */
	word8 b[4][4];
	int i, j;
		
	for(j = 0; j < BC; j++)
		for(i = 0; i < 4; i++)
			b[j][i] = mul(2,a[j][i])
				^ mul(3,a[j][(i + 1) % 4])
				^ a[j][(i + 2) % 4]
				^ a[j][(i + 3) % 4];
	for(i = 0; i < 4; i++)
		for(j = 0; j < BC; j++)
			a[j][i] = b[j][i];
}

static void InvMixColumn(word8 a[4][4], word8 BC) {
        /* Mix the four bytes of every column in a linear way
	 * This is the opposite operation of Mixcolumn
	 */
	int j;

	for(j = 0; j < BC; j++)
		*((word32*)a[j]) = *((word32*)U1[a[j][0]])
								^ *((word32*)U2[a[j][1]])
								^ *((word32*)U3[a[j][2]])
								^ *((word32*)U4[a[j][3]]);


}

int rijndaelKeySched (word8 k[MAXKC][4], int keyBits, word8 W[MAXROUNDS+1][4][4])
{
	/* Calculate the necessary round keys
	 * The number of calculations depends on keyBits and blockBits
	 */ 
	int j, r, t, rconpointer = 0;
	word8 tk[MAXKC][4];
	int KC = ROUNDS - 6;
	
	(void) keyBits;		/* Silence compiler warnings */

/*
 * Copy to tk array without requiring k to have word32 safe alignment
	for(j = KC-1; j >= 0; j--)
		*((word32*)tk[j]) = *((word32*)k[j]);
 */
	pgpCopyMemory( k, tk, KC*4 );
	r = 0;
	t = 0;
	/* copy values into round key array */
	for(j = 0; (j < KC) && (r < (ROUNDS+1)); ) {
		for (; (j < KC) && (t < 4); j++, t++)
			*((word32*)W[r][t]) = *((word32*)tk[j]);
		if (t == 4) {
			r++;
			t = 0;
		}
	}
		
	while (r < (ROUNDS+1)) { /* while not enough round key material calculated */
		/* calculate new values */
		tk[0][0] ^= S[tk[KC-1][1]];
		tk[0][1] ^= S[tk[KC-1][2]];
		tk[0][2] ^= S[tk[KC-1][3]];
		tk[0][3] ^= S[tk[KC-1][0]];
		tk[0][0] ^= rcon[rconpointer++];

		if (KC != 8)
			for(j = 1; j < KC; j++)
				*((word32*)tk[j]) ^= *((word32*)tk[j-1]);
		else {
			for(j = 1; j < KC/2; j++)
				*((word32*)tk[j]) ^= *((word32*)tk[j-1]);
			tk[KC/2][0] ^= S[tk[KC/2 - 1][0]];
			tk[KC/2][1] ^= S[tk[KC/2 - 1][1]];
			tk[KC/2][2] ^= S[tk[KC/2 - 1][2]];
			tk[KC/2][3] ^= S[tk[KC/2 - 1][3]];
			for(j = KC/2 + 1; j < KC; j++)
				*((word32*)tk[j]) ^= *((word32*)tk[j-1]);
		}
		/* copy values into round key array */
		for(j = 0; (j < KC) && (r < (ROUNDS+1)); ) {
			for (; (j < KC) && (t < 4); j++, t++)
				*((word32*)W[r][t]) = *((word32*)tk[j]);
			if (t == 4) {
				r++;
				t = 0;
			}
		}
	}		

	return 0;
}

static int rijndaelKeyEnctoDec (int keyBits, word8 W[MAXROUNDS+1][4][4])
{
	int r;

	(void) keyBits;		/* Silence compiler warnings */

	for (r = 1; r < ROUNDS; r++) {
		InvMixColumn(W[r], 4);
	}
	return 0;
}	

static int rijndaelKeyDectoEnc (int keyBits, word8 W[MAXROUNDS+1][4][4])
{
	int r;

	(void) keyBits;		/* Silence compiler warnings */

	for (r = 1; r < ROUNDS; r++) {
		MixColumn(W[r], 4);
	}
	return 0;
}	

static int rijndaelEncrypt (word8 a[16], word8 b[16],
							word8 rk[MAXROUNDS+1][4][4])
{
	/* Encryption of one block. 
	 */
	int r;
   word8 temp[4][4];

    *((word32*)temp[0]) = *((word32*)a) ^ *((word32*)rk[0][0]);
    *((word32*)temp[1]) = *((word32*)(a+4)) ^ *((word32*)rk[0][1]);
    *((word32*)temp[2]) = *((word32*)(a+8)) ^ *((word32*)rk[0][2]);
    *((word32*)temp[3]) = *((word32*)(a+12)) ^ *((word32*)rk[0][3]);
    *((word32*)b) = *((word32*)T1[temp[0][0]])
           ^ *((word32*)T2[temp[1][1]])
           ^ *((word32*)T3[temp[2][2]]) 
           ^ *((word32*)T4[temp[3][3]]);
    *((word32*)(b+4)) = *((word32*)T1[temp[1][0]])
           ^ *((word32*)T2[temp[2][1]])
           ^ *((word32*)T3[temp[3][2]]) 
           ^ *((word32*)T4[temp[0][3]]);
    *((word32*)(b+8)) = *((word32*)T1[temp[2][0]])
           ^ *((word32*)T2[temp[3][1]])
           ^ *((word32*)T3[temp[0][2]]) 
           ^ *((word32*)T4[temp[1][3]]);
    *((word32*)(b+12)) = *((word32*)T1[temp[3][0]])
           ^ *((word32*)T2[temp[0][1]])
           ^ *((word32*)T3[temp[1][2]]) 
           ^ *((word32*)T4[temp[2][3]]);
   for(r = 1; r < ROUNDS-1; r++) {
		*((word32*)temp[0]) = *((word32*)b) ^ *((word32*)rk[r][0]);
		*((word32*)temp[1]) = *((word32*)(b+4)) ^ *((word32*)rk[r][1]);
		*((word32*)temp[2]) = *((word32*)(b+8)) ^ *((word32*)rk[r][2]);
		*((word32*)temp[3]) = *((word32*)(b+12)) ^ *((word32*)rk[r][3]);
   *((word32*)b) = *((word32*)T1[temp[0][0]])
           ^ *((word32*)T2[temp[1][1]])
           ^ *((word32*)T3[temp[2][2]]) 
           ^ *((word32*)T4[temp[3][3]]);
   *((word32*)(b+4)) = *((word32*)T1[temp[1][0]])
           ^ *((word32*)T2[temp[2][1]])
           ^ *((word32*)T3[temp[3][2]]) 
           ^ *((word32*)T4[temp[0][3]]);
   *((word32*)(b+8)) = *((word32*)T1[temp[2][0]])
           ^ *((word32*)T2[temp[3][1]])
           ^ *((word32*)T3[temp[0][2]]) 
           ^ *((word32*)T4[temp[1][3]]);
   *((word32*)(b+12)) = *((word32*)T1[temp[3][0]])
           ^ *((word32*)T2[temp[0][1]])
           ^ *((word32*)T3[temp[1][2]]) 
           ^ *((word32*)T4[temp[2][3]]);
   }
   /* last round is special */   
	*((word32*)temp[0]) = *((word32*)b) ^ *((word32*)rk[ROUNDS-1][0]);
	*((word32*)temp[1]) = *((word32*)(b+4)) ^ *((word32*)rk[ROUNDS-1][1]);
	*((word32*)temp[2]) = *((word32*)(b+8)) ^ *((word32*)rk[ROUNDS-1][2]);
	*((word32*)temp[3]) = *((word32*)(b+12)) ^ *((word32*)rk[ROUNDS-1][3]);
   b[0] = T1[temp[0][0]][1];
   b[1] = T1[temp[1][1]][1];
   b[2] = T1[temp[2][2]][1]; 
   b[3] = T1[temp[3][3]][1];
   b[4] = T1[temp[1][0]][1];
   b[5] = T1[temp[2][1]][1];
   b[6] = T1[temp[3][2]][1]; 
   b[7] = T1[temp[0][3]][1];
   b[8] = T1[temp[2][0]][1];
   b[9] = T1[temp[3][1]][1];
   b[10] = T1[temp[0][2]][1]; 
   b[11] = T1[temp[1][3]][1];
   b[12] = T1[temp[3][0]][1];
   b[13] = T1[temp[0][1]][1];
   b[14] = T1[temp[1][2]][1]; 
   b[15] = T1[temp[2][3]][1];
	*((word32*)b) ^= *((word32*)rk[ROUNDS][0]);
	*((word32*)(b+4)) ^= *((word32*)rk[ROUNDS][1]);
	*((word32*)(b+8)) ^= *((word32*)rk[ROUNDS][2]);
	*((word32*)(b+12)) ^= *((word32*)rk[ROUNDS][3]);

	return 0;
}


static int rijndaelDecrypt (word8 a[16], word8 b[16],
							word8 rk[MAXROUNDS+1][4][4])
{
	int r;
   word8 temp[4][4];
	

    *((word32*)temp[0]) = *((word32*)a) ^ *((word32*)rk[ROUNDS][0]);
    *((word32*)temp[1]) = *((word32*)(a+4)) ^ *((word32*)rk[ROUNDS][1]);
    *((word32*)temp[2]) = *((word32*)(a+8)) ^ *((word32*)rk[ROUNDS][2]);
    *((word32*)temp[3]) = *((word32*)(a+12)) ^ *((word32*)rk[ROUNDS][3]);
    *((word32*)b) = *((word32*)T5[temp[0][0]])
           ^ *((word32*)T6[temp[3][1]])
           ^ *((word32*)T7[temp[2][2]]) 
           ^ *((word32*)T8[temp[1][3]]);
   *((word32*)(b+4)) = *((word32*)T5[temp[1][0]])
           ^ *((word32*)T6[temp[0][1]])
           ^ *((word32*)T7[temp[3][2]]) 
           ^ *((word32*)T8[temp[2][3]]);
   *((word32*)(b+8)) = *((word32*)T5[temp[2][0]])
           ^ *((word32*)T6[temp[1][1]])
           ^ *((word32*)T7[temp[0][2]]) 
           ^ *((word32*)T8[temp[3][3]]);
   *((word32*)(b+12)) = *((word32*)T5[temp[3][0]])
           ^ *((word32*)T6[temp[2][1]])
           ^ *((word32*)T7[temp[1][2]]) 
           ^ *((word32*)T8[temp[0][3]]);
   for(r = ROUNDS-1; r > 1; r--) {
		*((word32*)temp[0]) = *((word32*)b) ^ *((word32*)rk[r][0]);
		*((word32*)temp[1]) = *((word32*)(b+4)) ^ *((word32*)rk[r][1]);
		*((word32*)temp[2]) = *((word32*)(b+8)) ^ *((word32*)rk[r][2]);
		*((word32*)temp[3]) = *((word32*)(b+12)) ^ *((word32*)rk[r][3]);
		*((word32*)b) = *((word32*)T5[temp[0][0]])
           ^ *((word32*)T6[temp[3][1]])
           ^ *((word32*)T7[temp[2][2]]) 
           ^ *((word32*)T8[temp[1][3]]);
		*((word32*)(b+4)) = *((word32*)T5[temp[1][0]])
           ^ *((word32*)T6[temp[0][1]])
           ^ *((word32*)T7[temp[3][2]]) 
           ^ *((word32*)T8[temp[2][3]]);
		*((word32*)(b+8)) = *((word32*)T5[temp[2][0]])
           ^ *((word32*)T6[temp[1][1]])
           ^ *((word32*)T7[temp[0][2]]) 
           ^ *((word32*)T8[temp[3][3]]);
		*((word32*)(b+12)) = *((word32*)T5[temp[3][0]])
           ^ *((word32*)T6[temp[2][1]])
           ^ *((word32*)T7[temp[1][2]]) 
           ^ *((word32*)T8[temp[0][3]]);
   }
   /* last round is special */   
	*((word32*)temp[0]) = *((word32*)b) ^ *((word32*)rk[1][0]);
	*((word32*)temp[1]) = *((word32*)(b+4)) ^ *((word32*)rk[1][1]);
	*((word32*)temp[2]) = *((word32*)(b+8)) ^ *((word32*)rk[1][2]);
	*((word32*)temp[3]) = *((word32*)(b+12)) ^ *((word32*)rk[1][3]);
   b[0] = S5[temp[0][0]];
   b[1] = S5[temp[3][1]];
   b[2] = S5[temp[2][2]]; 
   b[3] = S5[temp[1][3]];
   b[4] = S5[temp[1][0]];
   b[5] = S5[temp[0][1]];
   b[6] = S5[temp[3][2]]; 
   b[7] = S5[temp[2][3]];
   b[8] = S5[temp[2][0]];
   b[9] = S5[temp[1][1]];
   b[10] = S5[temp[0][2]]; 
   b[11] = S5[temp[3][3]];
   b[12] = S5[temp[3][0]];
   b[13] = S5[temp[2][1]];
   b[14] = S5[temp[1][2]]; 
   b[15] = S5[temp[0][3]];
	*((word32*)b) ^= *((word32*)rk[0][0]);
	*((word32*)(b+4)) ^= *((word32*)rk[0][1]);
	*((word32*)(b+8)) ^= *((word32*)rk[0][2]);
	*((word32*)(b+12)) ^= *((word32*)rk[0][3]);

	return 0;
}



/*
 * Exported functions for PGP
 */

/*
 * Flags at end of priv array to record whether key schedule is in encrypt
 * or decrypt mode
 */
#define AES_ENCRYPTION_MODE	0x11
#define AES_DECRYPTION_MODE	0x22


static void
aes128Key(void *priv, void const *keymaterial)
{
	PGPSize privSize;
	ROUNDS = 128/32 + 6;
	privSize = 16*(ROUNDS+1);
	rijndaelKeySched( (void *)keymaterial, 128, priv );
	*((PGPByte *)priv + privSize) = AES_ENCRYPTION_MODE;
}

static void
aes192Key(void *priv, void const *keymaterial)
{
	PGPSize privSize;
	ROUNDS = 192/32 + 6;
	privSize = 16*(ROUNDS+1);
	rijndaelKeySched( (void *)keymaterial, 192, priv );
	*((PGPByte *)priv + privSize) = AES_ENCRYPTION_MODE;
}

static void
aes256Key(void *priv, void const *keymaterial)
{
	PGPSize privSize;
	ROUNDS = 256/32 + 6;
	privSize = 16*(ROUNDS+1);
	rijndaelKeySched( (void *)keymaterial, 256, priv );
	*((PGPByte *)priv + privSize) = AES_ENCRYPTION_MODE;
}


static void
aesEncrypt(void *priv, void const *in, void *out)
{
	PGPSize privSize = 16*(ROUNDS+1);
	/* Make sure key schedule is in the right mode */
	if (*((PGPByte *)priv + privSize) != AES_ENCRYPTION_MODE) {
		PGPInt32 keyBits = 32*(ROUNDS-6);
		rijndaelKeyDectoEnc( keyBits, priv );
		*((PGPByte *)priv + privSize) = AES_ENCRYPTION_MODE;
	}
	rijndaelEncrypt( (void *)in, out, priv );
}

static void
aesDecrypt(void *priv, void const *in, void *out)
{
	PGPSize privSize = 16*(ROUNDS+1);
	/* Make sure key schedule is in the right mode */
	if (*((PGPByte *)priv + privSize) != AES_DECRYPTION_MODE) {
		PGPInt32 keyBits = 32*(ROUNDS-6);
		rijndaelKeyEnctoDec( keyBits, priv );
		*((PGPByte *)priv + privSize) = AES_DECRYPTION_MODE;
	}
	rijndaelDecrypt( (void *)in, out, priv );
}



/*
 * Define a Cipher for the generic cipher.  This is the only
 * real exported thing -- everything else can be static, since everything
 * is referenced through function pointers!
 */
PGPCipherVTBL const cipherAES128 = {
	"AES128",
	kPGPCipherAlgorithm_AES128,
	16,			/* Blocksize */
	16,			/* Keysize */
	1 + 16*(128/32 + 7),	/* Scheduled key, 1 byte to flag enc/dec */
	alignof(PGPUInt32),
	aes128Key,
	aesEncrypt,
	aesDecrypt,
	NULL
};
PGPCipherVTBL const cipherAES192 = {
	"AES192",
	kPGPCipherAlgorithm_AES192,
	16,			/* Blocksize */
	24,			/* Keysize */
	1 + 16*(192/32 + 7),	/* Scheduled key, 1 byte to flag enc/dec */
	alignof(PGPUInt32),
	aes192Key,
	aesEncrypt,
	aesDecrypt,
	NULL
};
PGPCipherVTBL const cipherAES256 = {
	"AES256",
	kPGPCipherAlgorithm_AES256,
	16,			/* Blocksize */
	32,			/* Keysize */
	1 + 16*(256/32 + 7),	/* Scheduled key, 1 byte to flag enc/dec */
	alignof(PGPUInt32),
	aes256Key,
	aesEncrypt,
	aesDecrypt,
	NULL
};



#if UNITTEST




/* Test vectors, first line from each ECB known answer test */

/* 128 bit key */
PGPByte K1[] = {
	0x00, 0x01, 0x02, 0x03, 0x05, 0x06, 0x07, 0x08,
	0x0A, 0x0B, 0x0C, 0x0D, 0x0F, 0x10, 0x11, 0x12
};
PGPByte P1[] = {
	0x50, 0x68, 0x12, 0xA4, 0x5F, 0x08, 0xC8, 0x89,
	0xB9, 0x7F, 0x59, 0x80, 0x03, 0x8B, 0x83, 0x59
};
PGPByte C1[] = {
	0xD8, 0xF5, 0x32, 0x53, 0x82, 0x89, 0xEF, 0x7D,
	0x06, 0xB5, 0x06, 0xA4, 0xFD, 0x5B, 0xE9, 0xC9
};

/* 192 bit key */
PGPByte K2[] = {
	0x00, 0x01, 0x02, 0x03, 0x05, 0x06, 0x07, 0x08,
	0x0A, 0x0B, 0x0C, 0x0D, 0x0F, 0x10, 0x11, 0x12,
	0x14, 0x15, 0x16, 0x17, 0x19, 0x1A, 0x1B, 0x1C
};
PGPByte P2[] = {
	0x2D, 0x33, 0xEE, 0xF2, 0xC0, 0x43, 0x0A, 0x8A,
	0x9E, 0xBF, 0x45, 0xE8, 0x09, 0xC4, 0x0B, 0xB6
};
PGPByte C2[] = {
	0xDF, 0xF4, 0x94, 0x5E, 0x03, 0x36, 0xDF, 0x4C,
	0x1C, 0x56, 0xBC, 0x70, 0x0E, 0xFF, 0x83, 0x7F
};

/* 256 bit key */
PGPByte K3[] = {
	0x00, 0x01, 0x02, 0x03, 0x05, 0x06, 0x07, 0x08,
	0x0A, 0x0B, 0x0C, 0x0D, 0x0F, 0x10, 0x11, 0x12,
	0x14, 0x15, 0x16, 0x17, 0x19, 0x1A, 0x1B, 0x1C,
	0x1E, 0x1F, 0x20, 0x21, 0x23, 0x24, 0x25, 0x26
};
PGPByte P3[] = {
	0x83, 0x4E, 0xAD, 0xFC, 0xCA, 0xC7, 0xE1, 0xB3,
	0x06, 0x64, 0xB1, 0xAB, 0xA4, 0x48, 0x15, 0xAB
};
PGPByte C3[] = {
	0x19, 0x46, 0xDA, 0xBF, 0x6A, 0x03, 0xA2, 0xA2,
	0xC3, 0xD0, 0xB0, 0x50, 0x80, 0xAE, 0xD6, 0xFC
};



int
main(void)
{	/* Test driver for AES cipher */
	PGPByte priv[1 + 16*(256/32 + 7)]; /* size from cipherAES256 */
	PGPByte X[16], Y[16];

	aes128Key(priv, K1);
	aesEncrypt(priv, P1, X);
	if (memcmp(C1, X, sizeof(X)) == 0)
		printf ("Encryption test 1 passed\n");
	else
		printf ("ERROR ON ENCRYPTION TEST 1\n");
	aesDecrypt(priv, C1, Y);
	if (memcmp(P1, Y, sizeof(Y)) == 0)
		printf ("Decryption test 1 passed\n");
	else
		printf ("ERROR ON DECRYPTION TEST 1\n");

	aes192Key(priv, K2);
	aesEncrypt(priv, P2, X);
	if (memcmp(C2, X, sizeof(X)) == 0)
		printf ("Encryption test 2 passed\n");
	else
		printf ("ERROR ON ENCRYPTION TEST 2\n");
	aesDecrypt(priv, C2, Y);
	if (memcmp(P2, Y, sizeof(Y)) == 0)
		printf ("Decryption test 2 passed\n");
	else
		printf ("ERROR ON DECRYPTION TEST 2\n");

	aes256Key(priv, K3);
	aesEncrypt(priv, P3, X);
	if (memcmp(C3, X, sizeof(X)) == 0)
		printf ("Encryption test 3 passed\n");
	else
		printf ("ERROR ON ENCRYPTION TEST 3\n");
	aesDecrypt(priv, C3, Y);
	if (memcmp(P3, Y, sizeof(Y)) == 0)
		printf ("Decryption test 3 passed\n");
	else
		printf ("ERROR ON DECRYPTION TEST 3\n");

	return 0;	/* normal exit */
} /* main */

#endif /* UNITTEST */



#endif /* ] PGP_AES */


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
